--[[
	CameraZoomExtension.lua
	
	Autor: 		Ifko[nator]
	Datum: 		26.11.2024
	Version: 	5.0
	
	Changelog:	v1.0 @09.12.2016 - initial implementation in FS 17
				-------------------------------------------------------------------------
				v2.0 @01.12.2018 - convert to FS 19
				-------------------------------------------------------------------------
				V3.0 @27.11.2021 - convert to FS 22
				-------------------------------------------------------------------------
				V4.0 @21.04.2022 - added zoom function for the in door camera
								 - fix for Patch 1.4 and higher
				-------------------------------------------------------------------------
				V4.1 @01.09.2022 - added compatibility with the 'AnimatedIndoorParts.lua'
				-------------------------------------------------------------------------
				V5.0 @26.11.2024 - convert to FS 25
]]

CameraZoomExtension = {};

CameraZoomExtension.modDirectory = "";
CameraZoomExtension.currentModName = "";

for _, mod in pairs(g_modManager.mods) do
	if mod.title:upper() == "EXTENDED OUTDOOR CAMERA ZOOM" or mod.title:upper() == "ERWEITERTER KAMERA ZOOM" then		
		if g_modIsLoaded[tostring(mod.modName)] then	
			CameraZoomExtension.currentModName = mod.modName;
			CameraZoomExtension.modDirectory = mod.modDir

			break;
		end;
	end;
end;

local function getSpecByName(self, specName, currentModName)
    local spec = self["spec_" .. Utils.getNoNil(currentModName, CameraZoomExtension.currentModName) .. "." .. specName];

	if spec ~= nil then
        return spec;
    end;

    return self["spec_" .. specName];
end;

local function getL10nText(text)
	if text:sub(1, 6) == "$l10n_" then
		text = g_i18n:getText(text:sub(7), CameraZoomExtension.currentModName);
	elseif g_i18n:hasText(text, CameraZoomExtension.currentModName) then
		text = g_i18n:getText(text, CameraZoomExtension.currentModName);
	end;

	return text;
end;

function CameraZoomExtension.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Enterable, specializations);
end;

function CameraZoomExtension.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"onUpdate",
		"onRegisterActionEvents"
	};
	
	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, CameraZoomExtension);
	end;
end;

function CameraZoomExtension:onLoad(savegame)
	local specEnterable = getSpecByName(self, "enterable");
	local specCameraZoomExtension = getSpecByName(self, "cameraZoomExtension");
	local specAnimatedIndoorParts = getSpecByName(self, "animatedIndoorParts", self.customEnvironment);

	local modDesc = loadXMLFile("modDesc", CameraZoomExtension.modDirectory .. "modDesc.xml");

	specCameraZoomExtension.inputIsPressed = false;
	specCameraZoomExtension.resetIndoorCamera = false;
	
	specEnterable.camZoomMax = math.floor(Utils.getNoNil(getXMLFloat(modDesc, "modDesc.vehicleCam#transMax"), 60));
	specEnterable.camIndoorZoomMax = math.floor(Utils.getNoNil(getXMLFloat(modDesc, "modDesc.vehicleCam#indoorZoomMax"), 25));
	
	if specEnterable.cameras ~= nil then
		for cameraNumber, camera in pairs(specEnterable.cameras) do
			if camera.transMax ~= nil then
				if camera.transMax > 0 then
					if camera.transMax < specEnterable.camZoomMax then
						camera.transMax = specEnterable.camZoomMax;
					end;

					setFovY(camera.cameraNode, math.rad(44));
				end;
			end;

			if camera.isInside and specAnimatedIndoorParts == nil then
				camera.oldFovSaved = math.deg(getFovY(camera.cameraNode));
			end;
		end;
	end;

	delete(modDesc);
end;

function CameraZoomExtension:onRegisterActionEvents(isActiveForInput)
	local specCameraZoomExtension = getSpecByName(self, "cameraZoomExtension");
	
	if self.isClient then
		self:clearActionEventsTable(specCameraZoomExtension.actionEvents);
		local specAnimatedIndoorParts = getSpecByName(self, "animatedIndoorParts", self.customEnvironment);
        
		if self:getIsActiveForInput(true, true) and specAnimatedIndoorParts == nil then
            local actionEventId;
			local inputs = {
				"ZOOM_INDOOR_CAM_BUTTON",
				"RESET_INDOOR_CAM_BUTTON",
			};
			
			for _, input in pairs(inputs) do
				_, actionEventId = self:addActionEvent(specCameraZoomExtension.actionEvents, InputAction[input], self, CameraZoomExtension.zoomIndoorCamera, false, true, true, true, nil);

				g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL);
				g_inputBinding:setActionEventTextVisibility(actionEventId, true);
				g_inputBinding:setActionEventActive(actionEventId, true);
				g_inputBinding:setActionEventText(actionEventId, getL10nText("input_" .. input));
			end;
		end;
	end;
end;

function CameraZoomExtension.zoomIndoorCamera(self, actionName, inputValue, callbackState, isAnalog)
	local specCameraZoomExtension = getSpecByName(self, "cameraZoomExtension");

	specCameraZoomExtension.inputIsPressed = actionName == "ZOOM_INDOOR_CAM_BUTTON";
	specCameraZoomExtension.resetIndoorCamera = actionName == "RESET_INDOOR_CAM_BUTTON";
end;

function CameraZoomExtension:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	if self:getIsActiveForInput(true, true) then
		local specEnterable = getSpecByName(self, "enterable");
		local specCameraZoomExtension = getSpecByName(self, "cameraZoomExtension");
		local specAnimatedIndoorParts = getSpecByName(self, "animatedIndoorParts", self.customEnvironment);
		local camIsInside = false;

		if self.isClient and specEnterable.cameras ~= nil and specAnimatedIndoorParts == nil then
			for cameraNumber, camera in pairs(specEnterable.cameras) do
				if camera.oldFovSaved ~= nil then
					local allowUpdateFov = false;
					
					if camera.seatCam ~= nil and camera.seatCam ~= 0 then 
						local user = g_currentMission.userManager:getUserByUserId(self.spec_enterable.controllerUserId);

						if g_gameStateManager:getGameState() == GameState.PLAY 
							and self:getControllerName() == user:getNickname()
							and specEnterable.camIndex == specEnterable.seatCamIndex
						then
							local newFov = math.deg(getFovY(camera.seatCam));
							
							if specCameraZoomExtension.inputIsPressed then
								if newFov > specEnterable.camIndoorZoomMax then
									newFov = math.max(newFov - 0.055 * dt, specEnterable.camIndoorZoomMax);
								
									allowUpdateFov = true;
								end;
							else
								if newFov < camera.oldFovSaved then	
									newFov = math.min(newFov + 0.055 * dt, camera.oldFovSaved);
								
									allowUpdateFov = true;
								end;
							end;
						
							if allowUpdateFov then
								setFovY(camera.seatCam, math.rad(newFov));
							end;
						
							if specCameraZoomExtension.resetIndoorCamera then
								camera:resetCamera();
							end;
						
							camIsInside = true;
						end;
					else
						if camera.isInside and camera.isActivated then
							local newFov = math.deg(getFovY(camera.cameraNode));

							if specCameraZoomExtension.inputIsPressed then
								if newFov > specEnterable.camIndoorZoomMax then
									newFov = math.max(newFov - 0.055 * dt, specEnterable.camIndoorZoomMax);

									allowUpdateFov = true;
								end;
							else
								if newFov < camera.oldFovSaved then	
									newFov = math.min(newFov + 0.055 * dt, camera.oldFovSaved);

									allowUpdateFov = true;
								end;
							end;

							if allowUpdateFov then
								setFovY(camera.cameraNode, math.rad(newFov));
							end;

							if specCameraZoomExtension.resetIndoorCamera then
								camera:resetCamera();
							end;

							camIsInside = true;
						end;
					end;
				end;
			end;
		end;

		for _, input in pairs({"ZOOM_INDOOR_CAM_BUTTON", "RESET_INDOOR_CAM_BUTTON"}) do
			local inputButton = specCameraZoomExtension.actionEvents[InputAction[input]];

			if inputButton ~= nil then
				g_inputBinding:setActionEventTextVisibility(inputButton.actionEventId, camIsInside and specAnimatedIndoorParts == nil);
				g_inputBinding:setActionEventActive(inputButton.actionEventId, camIsInside and specAnimatedIndoorParts == nil);
			end;
		end;

		specCameraZoomExtension.inputIsPressed = false;
		specCameraZoomExtension.resetIndoorCamera = false;
	end;
end;